   	package mars.tools;
   	import javax.swing.*;
import javax.swing.border.Border;
import javax.swing.border.EmptyBorder;
import javax.swing.border.TitledBorder;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableColumnModel;

import java.awt.*;
   	import java.awt.event.*;
import java.util.Vector;

import mars.*;
	import mars.mips.hardware.AddressErrorException;
	import mars.mips.hardware.Memory;
	import mars.mips.hardware.RegisterFile;
	import mars.mips.instructions.BasicInstruction;
	import mars.mips.instructions.BasicInstructionFormat;
import mars.simulator.Simulator;
import mars.util.SystemIO;
import mars.venus.VenusUI;
	import mars.venus.RunAssembleAction;
	import mars.venus.RunStepAction;
	import mars.venus.ExecutePane;
	import mars.venus.GuiAction;

	
/*
Copyright (c) 2003-2006,  Pete Sanderson and Kenneth Vollmar

Developed by Pete Sanderson (psanderson@otterbein.edu)
and Kenneth Vollmar (kenvollmar@missouristate.edu)

Permission is hereby granted, free of charge, to any person obtaining 
a copy of this software and associated documentation files (the 
"Software"), to deal in the Software without restriction, including 
without limitation the rights to use, copy, modify, merge, publish, 
distribute, sublicense, and/or sell copies of the Software, and to 
permit persons to whom the Software is furnished to do so, subject 
to the following conditions:

The above copyright notice and this permission notice shall be 
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR 
ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF 
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION 
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

(MIT license, http://www.opensource.org/licenses/mit-license.html)
 */
	
   /**
	 * Custom 5 stage pipelined MIPS processor using MarsTools!
	 */
    public class PipelinedMipsSimulator extends AbstractMarsToolAndApplication {
    	
    	//variable Declaration
    	//declaration for the action button
    	private JButton Step, Assemble, Setting, Check, Initialise; 
    	private Action runAssembleAction, runStepAction, runSettingAction;
    	private VenusUI mainUI;
        private JToolBar toolbar;
        private JPanel resultpane;
        private JTable result;
        private JScrollPane tableScrollPane;
        private JTextArea testing;
        private DefaultTableModel dtm;
    	private Container displayPanel = this.getContentPane();
    	private static SettingOption settingObj = null;
    	private InitialRegisterMemoryValue regValue = null;
    	//Declaration for the information on the Mars tool
    	private static String heading =  "Pipelined Simulator and Branch Explainer";
		private static String version = " Version 1.0";
		//Declaration
		private static int lastAddress = 0;
		static ExecutePane executePane;
		private int IFr = 0, IDr = -1, EXr = -1, MEMr = -1, WBr = -1, nop = 0;
		private int cyclesNo = 1;
		private TableColumnModel columnModel;
		private static boolean nopF = false, nopD = false, nopE = false, nopM = false, nopW = false, flushOne = false, flushTwo = false, flushThree = false;
   	    private static String prediction = "";
   	/**
   	 * Simple constructor, likely used to run a stand-alone memory reference visualizer.
   	 * @param title String containing title for title bar
   	 * @param heading String containing text for heading shown in upper part of window.
   	 */
       public PipelinedMipsSimulator(String title, String heading) {
         super(title,heading);
      }
   	 
   	 /**
   	  *  Simple constructor, likely used by the MARS Tools menu mechanism
   	  */
       public PipelinedMipsSimulator() {
         super (heading+", "+version, heading);
      }
   
       /**
   	  *  Required method to return Tool name.  
   	  *  @return  Tool name.  MARS will display this in menu item.
   	  */
       public String getName() {
         return "Pipelined Simulator and Branch Explainer";
      }
   	
      /**
   	 *  Implementation of the inherited abstract method to build the main 
   	 *  display area of the GUI.  It will be placed in the CENTER area of a 
   	 *  BorderLayout.  The title is in the NORTH area, and the controls are 
   	 *  in the SOUTH area.
   	 */
       protected JComponent buildMainDisplayArea() {
    	   mainUI = Globals.getGui();
       	   this.createActionObjects();
       	   toolbar= this.setUpToolBar();
       	   this.setUpTestingGUI();
	       resultpane = new JPanel();
	 	   resultpane.setLayout(new BoxLayout(resultpane, BoxLayout.Y_AXIS));
	 	   resultpane.add(testing);
	 	   resultpane.add(this.tableScrollPane);
	 	   resultpane.setBorder(BorderFactory.createLineBorder(Color.black));
       	   displayPanel.add(toolbar, BorderLayout.NORTH);
       	   displayPanel.add(resultpane, BorderLayout.WEST);
       	   this.setResizable(true);
       	   return (JComponent) displayPanel;
      }
       
       protected void initializePostGUI(){
			JEditorPane editorPane = null;
			String html = "<div style=\"margin: 0px 10px 5px 10px;\"><h1><u>How to Use</u></h1><p>Step 1: Select the Simulator settings that you want to "
					+ "use in the Setting option by click on Setting Button.</p><p>Step 2: Open the MIPS code that you want to simulator in the Simulator.</p>"
					+ "<p>Step 3: Click on the Assemble Button <b>(In the Pipeline tool [not the main window]).</b></p><p>Step 4: Click on the 'Check Dependency' Button (A pop up window 'New corrected MIPS program codes' will appear)"
					+ ".</p><p>Step 5: Copy the MIPS code in the pop up window into the editor and save it (ctrl+s).</p><p>Step 6: Click on the Assemble Button <b>(In the Pipeline tool [not the main window])"
					+ ".</b></p><p>Step 7: Click on the Step Button <b>(In the Pipeline tool [not the main window])</b> to run the code one cycle at a time.</p></div>";
			editorPane = new JEditorPane("text/html", html);
			editorPane.setEditable(false);
			JFrame defaultSetting = new JFrame("User Instruction Guide");
			defaultSetting.getContentPane().add(editorPane, BorderLayout.CENTER);
			defaultSetting.pack();
			Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();
			int w = defaultSetting.getSize().width;
			int h = defaultSetting.getSize().height;
			int x = (dim.width - w) / 2;
			int y = (dim.height - h) / 2;
			defaultSetting.setLocation(x, y);
			defaultSetting.setResizable(false);
			defaultSetting.setVisible(true);
       }
       
       //action performed by the help button
       protected JComponent getHelpComponent() {
           final String helpContent = "This tool help student to visualize how the data flow through a 5 stage pipelined MIPS processor";
              						;
           JButton help = new JButton("Help");
           help.addActionListener(
                  new ActionListener() {
                     public void actionPerformed(ActionEvent e) {
                       JOptionPane.showMessageDialog(theWindow, helpContent);
                    }
                 });		
           return help;  
        }
       
       protected void reset(){
    	   IF.initInst = null;
    	   IF.fetchInst = null;
    	   ID.initInst = null;
    	   ID.fetchInst = null;
    	   EX.initInst = null;
    	   EX.fetchInst = null;
    	   MEM.initInst = null;
    	   MEM.fetchInst = null;
    	   WB.fetchInst = null;
    	   IF.pcInit = -99;
    	   IF.nextPc = -99;
    	   IFr = 0;
    	   IDr = -1;
    	   EXr = -1;
    	   MEMr = -1;
    	   WBr = -1;
    	   nop = 0;
    	   settingObj.exitButton.doClick();
    	   Globals.getGui().getMainPane().getExecutePane().getTextSegmentWindow().unhighlightAllSteps();
           Globals.getGui().getMainPane().getExecutePane().getTextSegmentWindow().setCodeHighlighting(true);
           dtm = (DefaultTableModel)result.getModel();
           dtm.setColumnCount(1);
           int j = result.getRowCount();
           while (j > 0) {
               this.dtm.removeRow(j - 1);
               --j;
           }
           cyclesNo = 1;
           tableScrollPane.repaint();
           tableScrollPane.revalidate();
    	   testing.setText("CPI:" + "\n" + "Steady state CPI:");
    	   prediction = "";
           Step.setEnabled(true);
           ActionEvent e = null;
           this.runAssembleAction.actionPerformed(e);
       }
       
       //create the tool bar on top
       private JToolBar setUpToolBar() {
           JToolBar toolBar = new JToolBar();
           Assemble = new JButton(runAssembleAction);
           Assemble.setText("");
           Step = new JButton(runStepAction);
           Step.setText("");
           Setting = new JButton(runSettingAction);
           Setting.setText("");
           Check = new JButton("Check Dependency");
           Check.addActionListener(new ActionListener(){
    		   public void actionPerformed(ActionEvent e) {
    			   DependencyCheck.checkDependency();
    		    }
    	   });
           Initialise = new JButton("Set Reg Value");
           Initialise.addActionListener(new ActionListener(){
    		   public void actionPerformed(ActionEvent e) {
    			   if(regValue == null){
    				   regValue = new InitialRegisterMemoryValue();
    				   regValue.displayWindow();
    			   }else{
    				   regValue.displayWindow();
    			   }
    		    }
    	   });
           toolBar.add(Assemble);
           toolBar.add(Step);
           toolBar.add(Setting);
           toolBar.add(Check);
           toolBar.add(Initialise);
           return toolBar;     
       }
       
       //For Testing
       private void setUpTestingGUI(){
    	   	this.dtm = new DefaultTableModel(new Object[0][], new String[]{"PC|Cycle"}){
    		   @Override
    		    public boolean isCellEditable(int row, int column) {
    		        return false;
    		    }

    		    public Class getColumnClass(int column) {
    		        return this.getValueAt(0, column).getClass();
    		    }

    		    public boolean getScrollableTracksViewportWidth() {
    		        if (PipelinedMipsSimulator.this.getPreferredSize().width < PipelinedMipsSimulator.this.getParent().getWidth()) {
    		            return true;
    		        }
    		        return false;
    		    }
    	   };
    	   result = new JTable(this.dtm);
    	   this.testing = new JTextArea("CPI:" + "\n" + "Steady state CPI:");
    	   result.setAutoResizeMode(0);
           result.setShowGrid(false);
           this.columnModel = result.getColumnModel();
    	   this.tableScrollPane = new JScrollPane(result);
       }
       
       //set action in the menu bar.
       private void createActionObjects() {
           Toolkit tk = Toolkit.getDefaultToolkit();
           Class cs = this.getClass();
           try{
               runAssembleAction = new RunAssembleActionNew("Assemble",  
                       new ImageIcon(tk.getImage(cs.getResource(Globals.imagesPath+"Assemble22.png"))),
   							  "Assemble the current file and clear breakpoints", new Integer(KeyEvent.VK_A),
   							  KeyStroke.getKeyStroke( KeyEvent.VK_F3, 0), 
   							  mainUI);			

               runStepAction = new RunFiveStageStepAction("Step", 
                       new ImageIcon(tk.getImage(cs.getResource(Globals.imagesPath+"StepForward22.png"))),
   							  "Run one step at a time", new Integer(KeyEvent.VK_T),
   							  KeyStroke.getKeyStroke( KeyEvent.VK_F7, 0),
   							  mainUI);
               runSettingAction = new RunSettingAction("Setting", 
            		   new ImageIcon(tk.getImage(cs.getResource(Globals.imagesPath+"Settings22.png"))),
            		   		  "Change Data and Control Dependency Settings", new Integer(KeyEvent.VK_S),
            		   		  KeyStroke.getKeyStroke( KeyEvent.VK_F5, 0),
            		   		  mainUI);
           }
           catch(Exception e){
               System.out.println("Internal Error: images folder not found, or other null pointer exception while creating Action objects");
               e.printStackTrace();
               System.exit(0);
           }
       }

       public static SettingOption getSettingObject(){
    	   return settingObj;
       }

       private class RunSettingAction extends GuiAction{
    	   
    	   protected RunSettingAction(String name, Icon icon, String descrip, Integer mnemonic, KeyStroke accel, 
    			   VenusUI gui){
    		   super(name, icon, descrip, mnemonic, accel, gui);
    	   }
    	   public void actionPerformed(ActionEvent e) {
    		   if(settingObj == null){
    			   settingObj = new SettingOption();
    			   settingObj.displayOption();
    		   }else{
    			   settingObj.displayOption();
    		   }
    	   }
       }
       
       private class RunFiveStageStepAction extends GuiAction{

    	   protected RunFiveStageStepAction(String name, Icon icon, String descrip, Integer mnemonic, KeyStroke accel, 
    			   VenusUI gui) {
    		   super(name, icon, descrip, mnemonic, accel, gui);
    		   // TODO Auto-generated constructor stub
    	   }
    	   public void actionPerformed(ActionEvent e) {
	    		   	IF.run();
		   			ID.run();
		   			EX.run();
		   			MEM.run();
		   			WB.run();
		   			//update gui with register value
		    		mainUI.getMainPane().getExecutePane().getRegistersWindow().updateRegisters();
		    		//update gui with memory value
		    		mainUI.getMainPane().getExecutePane().getDataSegmentWindow().updateValues();
		   			//Highlight the next Instruction in the main GUI panel
		   			Globals.getGui().getMainPane().getExecutePane().getTextSegmentWindow().highlightStepAtPC();
		   			//For updating table
		   			updateTable();
    		   }
    	   
       }
       
       private class RunAssembleActionNew extends RunAssembleAction{
	   		public RunAssembleActionNew(String name, Icon icon, String descrip, Integer mnemonic, KeyStroke accel, VenusUI gui) 
	   		{
	   			super(name, icon, descrip, mnemonic, accel, gui);
	   		}
	   		
	   		public void actionPerformed(ActionEvent e) 
	   		{
	   			IF.initInst = IF.fetchInst= null;
				ID.initInst= ID.fetchInst = null;
				EX.initInst = EX.fetchInst = null;
				MEM.initInst = MEM.fetchInst = null;
				WB.fetchInst = null;
				prediction = "";
				super.actionPerformed(e);
				lastAddress = ((ProgramStatement)(Globals.program.getMachineList().get(Globals.program.getMachineList().size()-1) )).getAddress();
				RegisterFile.setProgramCounter(Memory.textBaseAddress);
				if(settingObj == null){
	    			   settingObj = new SettingOption();
	    		}
				//set register value
				if(regValue != null){
					for(int i= 0; i < 27; i++){
						RegisterFile.updateRegister(i+1, Integer.parseInt(regValue.registersValue[i].getText()));
					}
				}
				mainUI.getMainPane().getExecutePane().getRegistersWindow().updateRegisters();
	   		}
   		}
       
       private void updateTable(){
           dtm = (DefaultTableModel)result.getModel();
    	   ImageIcon IFImg = new ImageIcon(this.getClass().getResource("/images/IF.png"));
    	   Image imageIF = IFImg.getImage();
           Image newimgIF = imageIF.getScaledInstance(30, 20, 4);
           IFImg = new ImageIcon(newimgIF);
    	   ImageIcon IDImg = new ImageIcon(this.getClass().getResource("/images/ID.png"));
    	   Image imageID = IDImg.getImage();
           Image newimgID = imageID.getScaledInstance(30, 20, 4);
           IDImg = new ImageIcon(newimgID);
    	   ImageIcon EXImg = new ImageIcon(this.getClass().getResource("/images/EX.png"));
    	   Image imageEX = EXImg.getImage();
           Image newimgEX = imageEX.getScaledInstance(30, 20, 4);
           EXImg = new ImageIcon(newimgEX);
    	   ImageIcon MEMImg = new ImageIcon(this.getClass().getResource("/images/MEM.png"));
    	   Image imageMEM = MEMImg.getImage();
           Image newimgMEM = imageMEM.getScaledInstance(30, 20, 4);
           MEMImg = new ImageIcon(newimgMEM);
    	   ImageIcon WBImg = new ImageIcon(this.getClass().getResource("/images/WB.png"));
    	   Image imageWB = WBImg.getImage();
           Image newimgWB = imageWB.getScaledInstance(30, 20, 4);
           WBImg = new ImageIcon(newimgWB);
           ImageIcon EM = new ImageIcon(this.getClass().getResource("/images/Empty.png"));
    	   Image imageEM = EM.getImage();
           Image newimgEM = imageEM.getScaledInstance(30, 20, 4);
           EM = new ImageIcon(newimgEM);
           ImageIcon S = new ImageIcon(this.getClass().getResource("/images/Stall.png"));
    	   Image imageS = S.getImage();
           Image newimgS = imageS.getScaledInstance(30, 20, 4);
           S = new ImageIcon(newimgS);
           ImageIcon Flush = new ImageIcon(this.getClass().getResource("/images/Flush.png"));
    	   Image imageFlush = Flush.getImage();
           Image newimgFlush = imageFlush.getScaledInstance(30, 20, 4);
           Flush = new ImageIcon(newimgFlush);
           
           Vector<Object> tempC = new Vector();
           //when the table has no row
           if(result.getRowCount() == 0){
        	   Object[][] row1 = { { new Integer(1) } };
        	   dtm.addRow(row1);
        	   dtm.addColumn(Integer.valueOf(cyclesNo), new Object[] { IFImg });
        	   dtm.setValueAt(Integer.toHexString(IF.fetchInst.getAddress()), 0, 0);
           }else{
        	   if (((IFr >= 0) || (IDr >= 0) || (EXr >= 0) || (MEMr >= 0) || (WBr >= 0)) && (IFr != WBr)){
        		   for (int r = 0; r < result.getRowCount(); r++)
        			   tempC.addElement(EM);
        		   dtm.addColumn(Integer.valueOf(cyclesNo), tempC);
        	   }
        	   //Display the PC for current Instruction
               if(IF.fetchInst != null){
            	   Vector<String> row = new Vector<String>();
            	   row.addElement(Integer.toHexString(IF.fetchInst.getAddress()));
            	   dtm.addRow(row);
               }
               if(IFr >= 0 && (IF.fetchInst != null)){
            	   dtm.setValueAt(IFImg, IFr, dtm.getColumnCount()-1);
               }
               if(IDr >= 0 && (ID.fetchInst != null)){
            	   if(ID.flushInst){
            		   dtm.setValueAt(Flush, IDr, dtm.getColumnCount()-1);
            	   }else{
                	   dtm.setValueAt(IDImg, IDr, dtm.getColumnCount()-1);
            	   }
               }
               if(EXr >= 0 && (EX.fetchInst != null)){
            	   if(EX.flushInst){
            		   dtm.setValueAt(Flush, EXr, dtm.getColumnCount()-1);
            	   }else{
            		   dtm.setValueAt(EXImg, EXr, dtm.getColumnCount()-1);
            	   }
               }
               if(MEMr >= 0 && (MEM.fetchInst != null)){
            	   if(MEM.flushInst){
            		   dtm.setValueAt(Flush, MEMr, dtm.getColumnCount()-1);
            	   }else{
            		   dtm.setValueAt(MEMImg, MEMr, dtm.getColumnCount()-1);
            	   }
               }
               if(WBr >= 0 && (WB.fetchInst != null)){
            	   if(WB.flushInst){
            		   dtm.setValueAt(Flush, WBr, dtm.getColumnCount()-1);
            		   nop++;
            	   }else{
            		   dtm.setValueAt(WBImg, WBr, dtm.getColumnCount()-1);
            	   }
               }
               
               if(nopW)
            	   dtm.setValueAt(EM, WBr, dtm.getColumnCount()-1);
               if(nopM)
            	   dtm.setValueAt(EM, MEMr, dtm.getColumnCount()-1);
               if(nopE)
            	   dtm.setValueAt(EM, EXr, dtm.getColumnCount()-1);
               if(nopD)
            	   dtm.setValueAt(EM, IDr, dtm.getColumnCount()-1);
               if(nopF){
            	   nop++;
            	   dtm.setValueAt(S, IFr, dtm.getColumnCount()-1);
               }
           }//end else
           nopW = false;
           nopM = false;
           nopE = false;
           nopD = false;
           nopF = false;
           WBr = MEMr;
           MEMr = EXr;
           EXr = IDr;
           IDr = IFr;
           if(IF.fetchInst != null){
        	   IFr++;
           }
           cyclesNo++;
           result.scrollRectToVisible(result.getCellRect(result.getRowCount() - 1, result.getColumnCount() - 1, true));
           this.columnModel.getColumn(0).setPreferredWidth(100);
           int i = 1;
           while (i < this.columnModel.getColumnCount()) {
               this.columnModel.getColumn(i).setPreferredWidth(30);
               ++i;
           }
           if(IFr == WBr){
        	   float cycle = dtm.getColumnCount() - 1;
        	   float instruction = dtm.getRowCount();
        	   if(settingObj.BranchPredictType == 3){
        		   prediction = "\n" + prediction;
        		   prediction += "Current State: " + settingObj.State;
        	   }
        	   if(settingObj.BranchPredictType == 2){
        		   prediction = "\n" + prediction;
        		   prediction += "Current State: " + settingObj.State;
        	   }
        	   if(settingObj.BranchPredictType == 1){
        		   prediction = "\n" + prediction;
        		   prediction = prediction.substring(0, prediction.length() - 1);
        	   }
        	   testing.setText("CPI:" + cycle/(instruction - nop) + "\nSteady State CPI:" + instruction/(instruction - nop) + prediction);
        	   Step.setEnabled(false);
           }
       }
       
       private static void pipelineRegisterUpdate(){
    	   //update to next instruction (IF.initInst is a holder for current Instruction to be pass to the next stage)
    	   IF.initInst = IF.fetchInst;
    	   ID.initInst = ID.fetchInst;
    	   EX.initInst = EX.fetchInst;
    	   MEM.initInst = MEM.fetchInst;
    	   
    	   //pass pc and flush value from IF to ID
    	   IF.pcInit = IF.nextPc;
    	   IF.flushInstInit = IF.flushInst;
    	   
    	   //pass decoded register value from ID to EXE
    	   ID.rdInit = ID.rd;
    	   ID.rsInit = ID.rs;
    	   ID.rtInit = ID.rt;
    	   ID.raInit = ID.ra;
    	   ID.raWBInit = ID.raWB;
    	   ID.writeToRegInit = ID.writeToReg;
    	   ID.writeToMemInit = ID.writeToMem;
    	   ID.readMemInit = ID.readMem;
    	   ID.immInit = ID.imm;
    	   ID.pcInit = ID.pc;
    	   ID.RegValRsInit = ID.RegValRs;
    	   ID.RegValRtInit = ID.RegValRt;
    	   ID.flushInstInit = ID.flushInst;
    	   
    	   //pass ALU result from EX stage to MEM stage, Destination register value
    	   EX.rtValueInit = EX.rtValue;
    	   EX.aluResultInit = EX.aluResult;
    	   EX.rdInit = EX.rd;
    	   EX.raInit = EX.ra;
    	   EX.pcInit = EX.pc;
    	   EX.raWBInit = EX.raWB;
    	   EX.writeToRegInit = EX.writeToReg;
    	   EX.writeToMemInit = EX.writeToMem;
    	   EX.readMemInit = EX.readMem;
    	   EX.flushInstInit = EX.flushInst;
    	   
    	   //pass MEM stage to WB stage Destination register value
    	   MEM.rdInit = MEM.rd;
    	   MEM.raInit = MEM.ra;
    	   MEM.raWBInit = MEM.raWB;
    	   MEM.writeToRegInit = MEM.writeToReg;
    	   MEM.writeDataInit = MEM.writeData;
    	   MEM.flushInstInit = MEM.flushInst;
       }
       
       private static int BranchPrediction(ProgramStatement fetchInst, int pc, long aluResult, int raWB, int stage){
    	   boolean Taken = false;
    	   switch(settingObj.BranchPredictType){
		   		case 0:	if(aluResult != 0){//if Branch is Taken (No Branch Prediction)
		   					RegisterFile.setProgramCounter((int) (pc + (aluResult << 2)));
		   					if(fetchInst.getInstruction().getName().equals("bgezal") || fetchInst.getInstruction().getName().equals("bltzal"))
		   						raWB = 1;
				   		}
		   				break;
		   		case 1:	if(settingObj.AlwaysBranchEnable){//if Branch Always Taken (Static Prediction)
		   					if((aluResult == 0)){//if Branch is actually Not Taken
		   						prediction += "[" + fetchInst.getSource().trim() + "] Prediction: Taken | Actual: Not Taken\n";
		   						if(stage == 3)
		   							flushOne = true;//IF.flushInst = true;
		   						if(stage == 4)
		   							flushTwo = true;
		   						if(stage != 2){
			   						RegisterFile.setProgramCounter((int) (pc));
			   						if(fetchInst.getInstruction().getName().equals("bgezal") || fetchInst.getInstruction().getName().equals("bltzal"))
				   						raWB = 0;
		   						}
		   					}else{//branch actually Taken does not need any action (for stage 3 and 4)
		   						prediction += "[" + fetchInst.getSource().trim() + "] Prediction: Taken | Actual: Taken\n";
		   						if(stage == 2){
		   							flushOne = true;
		   							RegisterFile.setProgramCounter((int) (pc + (aluResult << 2)));
				   					if(fetchInst.getInstruction().getName().equals("bgezal") || fetchInst.getInstruction().getName().equals("bltzal"))
				   						raWB = 1;
		   						}
		   					}
		   				}else{//Branch Always Not Taken (Static Prediction)
		   					if(aluResult != 0){//if Branch is actually Taken
		   						prediction += "[" + fetchInst.getSource().trim() + "] Prediction: Not Taken | Actual: Taken\n";
		   						if(stage == 2)
		   							flushOne = true;
		   						if(stage == 3)
		   							flushTwo = true;//IF.flushInst = true;
		   						if(stage == 4)
		   							flushThree = true;
		   						RegisterFile.setProgramCounter((int) (pc + (aluResult << 2)));
			   					if(fetchInst.getInstruction().getName().equals("bgezal") || fetchInst.getInstruction().getName().equals("bltzal"))
			   						raWB = 1;
		   					}else{//branch actually Not Taken does not need any action
		   						prediction += "[" + fetchInst.getSource().trim() + "] Prediction: Not Taken | Actual: Not Taken\n";
		   					}    						   					
		   				}
		   				break;
		   		case 2:	if(settingObj.AlwaysBranchEnable){//if state 2, 3 [Branch Always Taken](Dynamic Prediction)
		   					if((aluResult == 0)){//if Branch is actually Not Taken
		   						prediction += "[" + fetchInst.getSource().trim() + "] [State:" + settingObj.State + "] Prediction: Taken | Actual: Not Taken\n";
		   						if(stage == 3)	
		   							flushOne = true;//IF.flushInst = true;
		   						if(stage == 4)
		   							flushTwo = true;
		   						if(stage != 2){
			   						RegisterFile.setProgramCounter((int) (pc));
			   						if(fetchInst.getInstruction().getName().equals("bgezal") || fetchInst.getInstruction().getName().equals("bltzal"))
				   						raWB = 0;
		   						}
		   					}else{//branch actually Taken does not need any action
		   						Taken = true;
		   						prediction += "[" + fetchInst.getSource().trim() + "] [State:" + settingObj.State + "] Prediction: Taken | Actual: Taken\n";
		   						if(stage == 2){
		   							flushOne = true;
		   							RegisterFile.setProgramCounter((int) (pc + (aluResult << 2)));
				   					if(fetchInst.getInstruction().getName().equals("bgezal") || fetchInst.getInstruction().getName().equals("bltzal"))
				   						raWB = 1;
		   						}
		   					}
		   				}else{//if state 0, 1 [Branch Always Not Taken] (Dynamic Prediction)
		   					if(aluResult != 0){//if Branch is actually Taken
		   						prediction += "[" + fetchInst.getSource().trim() + "] [State:" + settingObj.State + "] Prediction: Not Taken | Actual: Taken\n";
		   						Taken = true;
		   						if(stage == 2)
		   							flushOne = true;
		   						if(stage == 3)
		   							flushTwo = true;//IF.flushInst = true;
		   						if(stage == 4)
		   							flushThree = true;
		   						RegisterFile.setProgramCounter((int) (pc + (aluResult << 2)));
			   					if(fetchInst.getInstruction().getName().equals("bgezal") || fetchInst.getInstruction().getName().equals("bltzal"))
			   						raWB = 1;
		   					}else{//branch actually Not Taken does not need any action
		   						prediction += "[" + fetchInst.getSource().trim() + "] [State:" + settingObj.State + "] Prediction: Not Taken | Actual: Not Taken\n";
		   					}							   					
		   				}
		   				if(Taken){//if Branch is taken 
		   					if(settingObj.State != 3)//if not state 3, then state + 1
		   						settingObj.State = settingObj.State + 1;
		   					if(settingObj.State > 1)//if state is 2 or 3, then AlwaysBranch
		   						settingObj.AlwaysBranchEnable = true;
		   				}else{//if Branch is not taken
		   					if(settingObj.State != 0)//if not state 0, then state - 1
		   						settingObj.State = settingObj.State - 1;
		   					if(settingObj.State < 2)//if state is 0 or 1, then AlwaysNotBranch
		   						settingObj.AlwaysBranchEnable = false;
		   				}
		   				break;
		   		case 3: if(settingObj.AlwaysBranchEnable){//if state 2, 3 [Branch Always Taken](Dynamic Prediction)
		   					if((aluResult == 0)){//if Branch is actually Not Taken
		   						prediction += "[" + fetchInst.getSource().trim() + "] [State:" + settingObj.State + "] Prediction: Taken | Actual: Not Taken\n";
		   						if(stage == 3)	
		   							flushOne = true;//IF.flushInst = true;
		   						if(stage == 4)
		   							flushTwo = true;
		   						if(stage != 2){
			   						RegisterFile.setProgramCounter((int) (pc));
			   						if(fetchInst.getInstruction().getName().equals("bgezal") || fetchInst.getInstruction().getName().equals("bltzal"))
				   						raWB = 0;
		   						}
		   					}else{//branch actually Taken does not need any action
		   						Taken = true;
		   						prediction += "[" + fetchInst.getSource().trim() + "] [State:" + settingObj.State + "] Prediction: Taken | Actual: Taken\n";
		   						if(stage == 2){
		   							flushOne = true;
		   							RegisterFile.setProgramCounter((int) (pc + (aluResult << 2)));
				   					if(fetchInst.getInstruction().getName().equals("bgezal") || fetchInst.getInstruction().getName().equals("bltzal"))
				   						raWB = 1;
		   						}
		   					}
		   				}else{//if state 0, 1 [Branch Always Not Taken] (Dynamic Prediction)
		   					if(aluResult != 0){//if Branch is actually Taken
		   						prediction += "[" + fetchInst.getSource().trim() + "] [State:" + settingObj.State + "] Prediction: Not Taken | Actual: Taken\n";
		   						Taken = true;
		   						if(stage == 2)
		   							flushOne = true;
		   						if(stage == 3)
		   							flushTwo = true;//IF.flushInst = true;
		   						if(stage == 4)
		   							flushThree = true;
		   						RegisterFile.setProgramCounter((int) (pc + (aluResult << 2)));
			   					if(fetchInst.getInstruction().getName().equals("bgezal") || fetchInst.getInstruction().getName().equals("bltzal"))
			   						raWB = 1;
		   					}else{//branch actually Not Taken does not need any action
		   						prediction += "[" + fetchInst.getSource().trim() + "] [State:" + settingObj.State + "] Prediction: Not Taken | Actual: Not Taken\n";
		   					}							   					
		   				}
		   				if(Taken){//if Branch is taken 
		   					if(settingObj.State != 1){//if not state 1, then state + 1
		   						settingObj.State = settingObj.State + 1;
		   						settingObj.AlwaysBranchEnable = true;
		   					}
		   				}else{//if Branch is not taken
		   					if(settingObj.State != 0){//if not state 0, then state - 1
		   						settingObj.State = settingObj.State - 1;
		   						settingObj.AlwaysBranchEnable = false;
		   					}
		   				}
		   				break;
    	   }//end switch
    	   return raWB;
       }
       
       private static class IF{
    	   public static ProgramStatement initInst =null , fetchInst=null;
    	   public static int nextPc, pcInit;
    	   public static boolean flushInst, flushInstInit;
    	   public IF(){
    		   RegisterFile.setProgramCounter(Memory.textBaseAddress);//set the PC to the start of program
    	   }
    	   
    	   private static void run(){
    		   if(flushOne){
    			   flushInst = true;
    			   flushOne = false;
    		   }
    		   if(flushTwo){
    			   flushInst = true;
    			   ID.flushInst = true;
    			   flushTwo = false;
    		   }
    		   if(flushThree){
    			   flushInst = true;
    			   ID.flushInst = true;
    			   EX.flushInst = true;
    			   flushThree = false;
    		   }
    		   pipelineRegisterUpdate();
    		   try{
    				//check if not the last instruction.
   					if(RegisterFile.getProgramCounter() > lastAddress){
	   					fetchInst = null;
	   					return;
	   				}
	   				fetchInst = Globals.memory.getStatement( RegisterFile.getProgramCounter() );
	   				flushInst = false;
	   				nextPc = RegisterFile.getProgramCounter()+ 4;
	   				RegisterFile.setProgramCounter(nextPc);
	   				if(fetchInst.getInstruction().getName().equals("nop"))
	   					nopF = true;
	   			}
	   			catch(AddressErrorException e) 
	   			{
	   				//send message of program counter error
	   			}
    	   }
       }
       
       private static class ID{
    	   public static ProgramStatement initInst =null , fetchInst=null;
    	   public static int rs, rt, rd, ra, imm, pc;
    	   public static int raWB, writeToReg, writeToMem, readMem;
    	   public static int rsInit, rtInit, rdInit, raInit, immInit, pcInit;
    	   public static int raWBInit, writeToRegInit, writeToMemInit, readMemInit;
    	   public static int RegValRs, RegValRt;
    	   public static int RegValRsInit, RegValRtInit;
    	   public static long aluResult;
    	   public static boolean flushInst, flushInstInit;
    	   private static BasicInstruction instr = null;
    	   public ID(){
    		   return;
    	   }
    	   
    	   private static void run(){
    		   //initial rs, rt, rd, ra, imm
    		   rs = rt = ra = imm = 0;
    		   rd = -1;
    		   //initial control signal
    		   raWB = writeToReg = writeToMem = readMem = 0;
    		   fetchInst = IF.initInst;
    		   pc = IF.pcInit;
    		   flushInst = IF.flushInstInit;
    		   if(fetchInst == null)
    			   return;
    		   if(flushInst)
    			   return;
    		   if(fetchInst.getInstruction().getName().equals("nop")){
  					nopD = true;
  					return;
    		   }
    		   instr = (BasicInstruction) fetchInst.getInstruction();
    		   if( instr.getInstructionFormat() == BasicInstructionFormat.I_FORMAT){
//    			   rd = fetchInst.getOperand(0);
//    			   imm = fetchInst.getOperand(1);
//    			   rs = fetchInst.getOperand(2);
    			   //I-Format decode using self created code
    			   rs = Integer.parseInt(fetchInst.getMachineStatement().substring(6, 11), 2);
    			   rd = Integer.parseInt(fetchInst.getMachineStatement().substring(11, 16), 2);
    			   imm = Integer.parseInt(fetchInst.getMachineStatement().substring(16, 32), 2);
    			   RegValRs = RegisterFile.getValue(rs);
    			   writeToReg = 1;
    			   if (instr.getName().equals("sw") || instr.getName().equals("sb")){
    				   writeToMem = 1;
    				   writeToReg = 0;
    				   rt = rd;
    				   rd = -1;
        			   RegValRt = RegisterFile.getValue(rt);
    			   }
    			   if(instr.getName().equals("lw")){
    				   readMem = 1;
    			   }
    		   } else if(instr.getInstructionFormat() == BasicInstructionFormat.J_FORMAT){
//    			   imm = fetchInst.getOperand(0);
    			   //J-Format decode using self created code
    			   imm = Integer.parseInt(fetchInst.getMachineStatement().substring(6, 32), 2);
    			   //update the PC immediate at the decode stage(for J only)
    			   switch(fetchInst.getInstruction().getName()){
    			   		case "jal": ra = RegisterFile.getProgramCounter();
    			   					raWB = 1; //control signal
    			   		case "j":	if(settingObj.JumpStage == 2){
    			   						RegisterFile.setProgramCounter((RegisterFile.getProgramCounter() & 0xF0000000) | (imm << 2));
    			   					}
    			   					break;
    			   }
    		   } else if(instr.getInstructionFormat() == BasicInstructionFormat.R_FORMAT){
//    			   rd = fetchInst.getOperand(0);
//    			   rs = fetchInst.getOperand(1);
//    			   rt = fetchInst.getOperand(2);
    			   //R-Format decode using self created code
    			   rs = Integer.parseInt(fetchInst.getMachineStatement().substring(6, 11), 2);
    			   rt = Integer.parseInt(fetchInst.getMachineStatement().substring(11, 16), 2);
    			   rd = Integer.parseInt(fetchInst.getMachineStatement().substring(16, 21), 2);
    			   if(fetchInst.getInstruction().getName().equals("mthi")){
    				   rd = 33;
    			   }
    			   if(fetchInst.getInstruction().getName().equals("mtlo")){
    				   rd = 34;
    			   }
    			   if(fetchInst.getInstruction().getName().equals("jr")){
    				   RegisterFile.setProgramCounter(RegisterFile.getValue(rs));
    			   }
    			   RegValRs = RegisterFile.getValue(rs);
    			   RegValRt = RegisterFile.getValue(rt);
    			   writeToReg = 1;
    		   } else if(instr.getInstructionFormat() == BasicInstructionFormat.I_BRANCH_FORMAT){
//    			   rt = fetchInst.getOperand(0);
//    			   rs = fetchInst.getOperand(1);
//    			   imm = fetchInst.getOperand(2);
    			   rs = Integer.parseInt(fetchInst.getMachineStatement().substring(6, 11), 2);
    			   rt = Integer.parseInt(fetchInst.getMachineStatement().substring(11, 16), 2);
    			   imm = Integer.parseInt(fetchInst.getMachineStatement().substring(16, 32), 2);
    			   RegValRs = RegisterFile.getValue(rs);
    			   RegValRt = RegisterFile.getValue(rt);
    			   switch(fetchInst.getInstruction().getName()){
    			   		case "bgezal":
    			   		case "bltzal": 	ra = RegisterFile.getProgramCounter();
    			   }
    			   if(settingObj.BranchStage == 2){//Early Branch Evaluation
    				   if(settingObj.DataForwardingEnable){//DataForwarding is Enable
	    				   if(rs == MEM.rdInit){
	    					   RegValRs = (int) MEM.writeDataInit;
						   }
						   if( (rs == EX.rdInit) && (readMemInit != 1)){
							   RegValRs = (int) EX.aluResultInit;
						   }
						   if(rt == MEM.rdInit){
							   RegValRt = (int) MEM.writeDataInit;
						   }
						   if( (rt == EX.rdInit) && (readMemInit != 1)){
							   RegValRt = (int) EX.aluResultInit;
						   }
    				   }else if(settingObj.WBDECSimultEnable){
    					   if(rs == MEM.rdInit){
	    					   RegValRs = (int) MEM.writeDataInit;
						   }
    					   if(rt == MEM.rdInit){
							   RegValRt = (int) MEM.writeDataInit;
						   }
    				   }
					   try{
						   aluResult = instr.getSimulationCode().simulatePipeline(RegValRs, RegValRt, imm, fetchInst);
					   }catch(ProcessingException pe){
			    			   //do something
					   }
					   raWB = BranchPrediction(fetchInst, pc, aluResult, raWB, 2);
    			   }else if(settingObj.BranchPredictType == 1 || settingObj.BranchPredictType == 2){//Early Branch Prediction(Static Prediction and Dynamic Prediction)
    				   if(settingObj.AlwaysBranchEnable){//Always Branch(Static Prediction) | state 2 or 3 (Dynamic Prediction)
    					   flushOne = true;
    					   RegisterFile.setProgramCounter((int) (pc + (imm << 2)));
    					   if(fetchInst.getInstruction().getName().equals("bgezal") || fetchInst.getInstruction().getName().equals("bltzal"))
							   raWB = 1;
    				   }
    			   }
    		   }
    	   }
       }
       
       private static class EX{
    	   public static ProgramStatement initInst =null , fetchInst=null;
    	   public static long aluResult, aluResultInit;
    	   public static int rd, ra, rdInit, raInit, pc, pcInit;
    	   public static int raWB, writeToReg, writeToMem, readMem;
    	   public static int raWBInit, writeToRegInit, writeToMemInit, readMemInit;
    	   public static int rsValue, rtValue, immValue, rtValueInit;
    	   public static int rs, rt, rtInit;
    	   public static boolean flushInst, flushInstInit;
    	   private static BasicInstruction instr = null;
    	   public EX(){
    		   return;
    	   }
    	   
    	   private static void run(){
    		   //initial rs, rt, rd, ra, imm
    		   rs = rt = ra = immValue = 0;
    		   rd = -1;
    		   aluResult = 0;
    		   //initial control signal
    		   raWB = writeToReg = writeToMem = readMem = 0;
    		   fetchInst = ID.initInst;
    		   flushInst = ID.flushInstInit;
    		   if(fetchInst == null)
    			   return;
    		   if(flushInst)
    			   return;
    		   if(fetchInst.getInstruction().getName().equals("nop")){
  					nopE = true;
  					return;
    		   }
    		   rs = ID.rsInit;
    		   rt = ID.rtInit;
    		   rd = ID.rdInit;
    		   ra = ID.raInit;
    		   raWB = ID.raWBInit;
    		   pc = ID.pcInit;
    		   writeToReg = ID.writeToRegInit;
    		   writeToMem = ID.writeToMemInit;
    		   readMem = ID.readMemInit;
    		   rsValue = ID.RegValRsInit;
    		   rtValue = ID.RegValRtInit;
    		   immValue = ID.immInit;
			   instr = (BasicInstruction) fetchInst.getInstruction();
    		   try{
    			   //Enable Data Forwarding
    			   if(settingObj.DataForwardingEnable){
    				   if( instr.getInstructionFormat() == BasicInstructionFormat.I_FORMAT){
    					   rsValue = RegisterFile.getValue(rs);
    					   //check if rs match any not updated rd(from MEM and WB)
    					   if(rs == MEM.rdInit){
    						   rsValue = (int) MEM.writeDataInit;
    					   }
    					   if( (rs == rdInit) && (readMemInit != 1) ){
    						   rsValue = (int) aluResultInit;
    					   }
    					   if(fetchInst.getInstruction().getName().equals("sw") || fetchInst.getInstruction().getName().equals("sb")){
    						   if(rt == MEM.rdInit){
        						   rtValue = (int) MEM.writeDataInit;
        					   }
        					   if( (rt == rdInit) && (readMemInit != 1) ){
        						   rtValue = (int) aluResultInit;
        					   } 
    					   }
    					   aluResult = instr.getSimulationCode().simulatePipeline(rsValue, rtValue, immValue, fetchInst);
    				   } else if(instr.getInstructionFormat() == BasicInstructionFormat.R_FORMAT){
    					   rsValue = RegisterFile.getValue(rs);
    					   rtValue = RegisterFile.getValue(rt);
    					   //check if rs and rt match any not updated rd(from MEM and WB)
    					   //Latest Result is checked last to ensure latest result is used instead
    					   if(rs == MEM.rdInit){
    						   rsValue = (int) MEM.writeDataInit;
    					   }
    					   if( (rs == rdInit)  && (readMemInit != 1)){
    						   rsValue = (int) aluResultInit;
    					   }
    					   if(rt == MEM.rdInit){
    						   rtValue = (int) MEM.writeDataInit;
    					   }
    					   if( (rt == rdInit)  && (readMemInit != 1)){
    						   rtValue = (int) aluResultInit;
    					   }
    					   aluResult = instr.getSimulationCode().simulatePipeline(rsValue, rtValue, immValue, fetchInst);
    					   if(fetchInst.getInstruction().getName().equals("mflo")){
    						   if(MEM.rdInit == 34){
    							   aluResult = MEM.writeDataInit;
    						   }
    						   if((rdInit == 34) && (readMemInit != 1)){
    							   aluResult = aluResultInit;
    						   }
    					   }
    					   if(fetchInst.getInstruction().getName().equals("mfhi")){
    						   if(MEM.rdInit == 33){
    							   aluResult = MEM.writeDataInit;
    						   }
    						   if((rdInit == 33) && (readMemInit != 1)){
    							   aluResult = aluResultInit;
    						   }
    					   }
    				   } else if(instr.getInstructionFormat() == BasicInstructionFormat.I_BRANCH_FORMAT){
    					   rsValue = RegisterFile.getValue(rs);
    					   rtValue = RegisterFile.getValue(rt);
    					   //check if rs and rt match any not updated rd(from MEM and WB)
    					 //Latest Result is checked last to ensure latest result is used instead
    					   if(rs == MEM.rdInit){
    						   rsValue = (int) MEM.writeDataInit;
    					   }
    					   if( (rs == rdInit) && (readMemInit != 1)){
    						   rsValue = (int) aluResultInit;
    					   }
    					   if(rt == MEM.rdInit){
    						   rtValue = (int) MEM.writeDataInit;
    					   }
    					   if( (rt == rdInit) && (readMemInit != 1)){
    						   rtValue = (int) aluResultInit;
    					   }
    					   aluResult = instr.getSimulationCode().simulatePipeline(rsValue, rtValue, immValue, fetchInst);
    					   if(settingObj.BranchStage == 3){//Data forwarding case of Branch at EX stage
    						   raWB = BranchPrediction(fetchInst, pc, aluResult, raWB, 3);
    					   }
    				   }
    				   
    			   }else{//end Data Forwarding
    				   //Enable Simultaneous DEC and WB (Need 2 Nop) or Disable Simultaneous DEC and WB (Need 3 Nop)
    				   aluResult = instr.getSimulationCode().simulatePipeline(rsValue, rtValue, immValue, fetchInst);
    				   if((settingObj.BranchStage == 3) && (instr.getInstructionFormat() == BasicInstructionFormat.I_BRANCH_FORMAT)){//Enable Simultaneous DEC and WB case of Branch at EX stage
    					   raWB = BranchPrediction(fetchInst, pc, aluResult, raWB, 3);
					   }
    			   }
    			   if(settingObj.JumpStage == 3 && (instr.getInstructionFormat() == BasicInstructionFormat.J_FORMAT)){
    				   RegisterFile.setProgramCounter((RegisterFile.getProgramCounter() & 0xF0000000) | (immValue << 2));
    			   }
    		   }
    		   catch(ProcessingException pe){
    			   //do something
    		   }
    	   }
       }
       
       private static class MEM{
    	   public static ProgramStatement initInst =null , fetchInst=null; 	 
    	   public static long writeData, writeDataInit;
    	   public static int rd, rtValue, ra, rdInit, raInit, pc;
    	   public static int raWB, writeToReg, writeToMem, readMem;
    	   public static int raWBInit, writeToRegInit;
    	   public static boolean flushInst, flushInstInit;
    	   private static BasicInstruction instr = null;
    	   public MEM(){
    		   return;
    	   }  
    	   
    	   private static void run(){
    		   //initial rd, ra
    		   ra = 0;
    		   rd = -1;
    		   //initial control signal
    		   raWB = writeToReg = writeToMem = readMem = 0;
    		   fetchInst = EX.initInst;
    		   flushInst = EX.flushInstInit;
    		   if(fetchInst == null)
    			   return;
    		   if(flushInst)
    			   return;
    		   if(fetchInst.getInstruction().getName().equals("nop")){
  					nopM = true;
  					return;
    		   }
    		   writeData = EX.aluResultInit;
    		   rd = EX.rdInit;
    		   ra = EX.raInit;
    		   rtValue = EX.rtValueInit;
    		   raWB = EX.raWBInit;
    		   pc = EX.pcInit;
    		   writeToReg = EX.writeToRegInit;
    		   writeToMem = EX.writeToMemInit;
    		   readMem = EX.readMemInit;
    		   instr = (BasicInstruction) fetchInst.getInstruction();
    		   if(readMem == 1){
	    		   if(fetchInst.getInstruction().getName().equals("lw")){
	    			   try{
	        			   writeData = Globals.memory.getWord((int) writeData);
	    			   }
	    			   catch(AddressErrorException pe){
	    				   pe.printStackTrace();
	    			   }
	    		   }
	    		   if(fetchInst.getInstruction().getName().equals("lb")){
	    			   try{
	        			   writeData = Globals.memory.getByte((int) writeData);
	    			   }
	    			   catch(AddressErrorException pe){
	    				   pe.printStackTrace();
	    			   }
	    		   }
    		   }
    		   if(instr.getInstructionFormat() == BasicInstructionFormat.I_BRANCH_FORMAT){
    			   //if writeData is not 0 meaning that the condition to branch is met
    			   if(settingObj.BranchStage == 4){//Branch at MEM stage
    				   raWB = BranchPrediction(fetchInst, pc, writeData, raWB, 4);
    			   }
    			   return;
    		   }
    		   if(writeToMem == 1){
    			   if(fetchInst.getInstruction().getName().equals("sw")){
	    			   try{
	    				   Globals.memory.setWord((int) writeData, rtValue);
	    			   }
	    			   catch(AddressErrorException pe){
	    				   pe.printStackTrace();
	    			   }
    			   }
    			   if(fetchInst.getInstruction().getName().equals("sb")){
	    			   try{
	    				   Globals.memory.setByte((int) writeData, (rtValue & 0x000000ff));
	    			   }
	    			   catch(AddressErrorException pe){
	    				   pe.printStackTrace();
	    			   }
    			   }
    		   }
    	   }
       }
       
       private static class WB{
    	   public static ProgramStatement fetchInst = null;
    	   public static int rd, ra, raWB, writeToReg;
    	   public static long writeData;
    	   public static boolean flushInst;
    	   public WB(){
    		   return;
    	   }
    	   
    	   private static void run(){
    		   fetchInst = MEM.initInst;
    		   flushInst = MEM.flushInstInit;
    		   if(fetchInst == null)
    			   return;
    		   if(flushInst)
    			   return;
    		   if(fetchInst.getInstruction().getName().equals("nop")){
  					nopW = true;
  					return;
    		   }
    		   rd = MEM.rdInit;
    		   ra = MEM.raInit;
    		   raWB = MEM.raWBInit;
    		   writeToReg = MEM.writeToRegInit;
    		   writeData = MEM.writeDataInit;
    		   if(raWB == 1){//write to $ra(31) register the return pc value(ra)
    			   RegisterFile.updateRegister(31, ra);
    		   }else if(writeToReg == 1){
    			   if(fetchInst.getInstruction().getName().equals("mult") || fetchInst.getInstruction().getName().equals("multu")
    					   || fetchInst.getInstruction().getName().equals("mul")){
    				   // Register 33 is HIGH and 34 is LOW
    				   RegisterFile.updateRegister(33, (int) (writeData >> 32));
    				   RegisterFile.updateRegister(34, (int) ((writeData << 32) >> 32));
    				   if(fetchInst.getInstruction().getName().equals("mul")){
    					   RegisterFile.updateRegister(rd, (int)((writeData << 32) >> 32));
    				   }
    			   }else if(fetchInst.getInstruction().getName().equals("div") || fetchInst.getInstruction().getName().equals("divu")){
    				   // Register 33 is HIGH and 34 is LOW
    				   RegisterFile.updateRegister(33, (int) ((writeData << 32) >> 32));
    				   RegisterFile.updateRegister(34, (int) (writeData >> 32));
    			   }else{
            		   RegisterFile.updateRegister(rd, (int) writeData);
    			   }
    			   if(settingObj.WBDECSimultEnable){
    				   if(ID.rs == rd){
        				   ID.RegValRs = (int) writeData;
    				   }
    				   if(ID.rt == rd){
    					   ID.RegValRt = (int) writeData;
    				   }
    			   }
    		   }
    	   }
       }

   	
   }